/*
 * Decompiled with CFR 0.152.
 */
package net.savignano.snotify.atlassian.common.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import net.savignano.snotify.atlassian.common.enums.EKeyPurpose;
import net.savignano.snotify.atlassian.common.util.KeyStoreContentFetcher;
import net.savignano.snotify.atlassian.common.util.SecurityUtil;
import net.savignano.thirdparty.org.bouncycastle.asn1.ASN1ObjectIdentifier;
import net.savignano.thirdparty.org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.RDN;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.X500Name;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.style.BCStyle;
import net.savignano.thirdparty.org.bouncycastle.cert.X509CertificateHolder;
import net.savignano.thirdparty.org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSAlgorithm;
import net.savignano.thirdparty.org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SmimeUtil {
    public static final String X509_FACTORY_KEY = "X.509";
    private static final Logger log = LoggerFactory.getLogger(SmimeUtil.class);

    public static final Predicate<X509Certificate> createPredicateForEmail(String email) {
        return c -> {
            try {
                return Arrays.asList(SmimeUtil.getEmails(c)).contains(email);
            }
            catch (CertificateEncodingException | CertificateParsingException e) {
                log.error(e.getMessage(), (Throwable)e);
                return false;
            }
        };
    }

    public static final String getSerialNumber(X509Certificate certificate) {
        if (certificate == null) {
            log.debug("Null value given for certificate.");
            return "<N/A>";
        }
        BigInteger serialNumber = certificate.getSerialNumber();
        if (serialNumber == null) {
            log.warn("No serial number in certificate. Might be corrupt. Certificate: {}", (Object)certificate);
            return "<N/A>";
        }
        return SmimeUtil.getSerialNumber(serialNumber);
    }

    public static final String getSerialNumber(BigInteger serialNumber) {
        if (serialNumber == null) {
            log.debug("Null value given for serial number.");
            return "<N/A>";
        }
        return serialNumber.toString(10) + " (0x" + serialNumber.toString(16) + ")";
    }

    public static final boolean isSupportedSymmetricKeyAlgorithm(ASN1ObjectIdentifier algorithm) {
        if (algorithm == null) {
            return false;
        }
        return !algorithm.equals(CMSAlgorithm.DES_CBC);
    }

    public static final boolean isSupportedAsymmetricKeyAlgorithm(ASN1ObjectIdentifier algorithm) {
        return algorithm != null;
    }

    public static final X509Certificate createCertificate(byte[] content) throws CertificateException {
        ByteArrayInputStream is = new ByteArrayInputStream(content);
        CertificateFactory fact = CertificateFactory.getInstance(X509_FACTORY_KEY);
        return (X509Certificate)fact.generateCertificate(is);
    }

    public static final X509Certificate getValidCertForEmail(KeyStore keyStore, String email) throws KeyStoreException, CertificateException {
        return SmimeUtil.getValidCertForEmail(keyStore, email, EKeyPurpose.UNDEFINED);
    }

    public static final X509Certificate getValidCertForEmail(KeyStore keyStore, String email, EKeyPurpose purpose) throws KeyStoreException, CertificateException {
        if (keyStore == null || email == null || purpose == null) {
            return null;
        }
        log.debug("Looking for valid certificate for email <{}> and purpose {}.", (Object)email, (Object)purpose);
        Predicate<X509Certificate> predEmail = SmimeUtil.createPredicateForEmail(email);
        Predicate<X509Certificate> predPurpose = c -> purpose == EKeyPurpose.UNDEFINED || SmimeUtil.isCertForPurpose(c, purpose);
        Predicate<X509Certificate> predValidity = c -> {
            try {
                c.checkValidity();
                log.trace("Found valid certificate for email <{}>. Serial number: {}", (Object)email, (Object)SmimeUtil.getSerialNumber(c));
                return true;
            }
            catch (CertificateExpiredException | CertificateNotYetValidException e) {
                log.trace("Found invalid certificate for email <{}>: {}", (Object)email, (Object)SmimeUtil.getSerialNumber(c));
                return false;
            }
        };
        KeyStoreContentFetcher fetcher = new KeyStoreContentFetcher(keyStore);
        Set<X509Certificate> certs = fetcher.getCertsFor(predEmail.and(predPurpose).and(predValidity));
        fetcher.destroy();
        Optional cert = certs.stream().reduce((c1, c2) -> c1.getNotAfter().after(c2.getNotAfter()) ? c1 : c2);
        if (cert.isPresent()) {
            log.debug("Found valid certificate for email <{}>: ", (Object)email, (Object)SmimeUtil.getSerialNumber((X509Certificate)cert.get()));
            log.trace("Certificate: {}", cert);
            return (X509Certificate)cert.get();
        }
        log.debug("Found no valid certificate for email <{}>.", (Object)email);
        return null;
    }

    public static final PrivateKey getPrivateKeyForCert(KeyStore keyStore, X509Certificate cert, char[] password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException {
        if (keyStore == null || cert == null) {
            return null;
        }
        String serialNumber = SmimeUtil.getSerialNumber(cert);
        log.debug("Getting private key for certificate with serial number: {}", (Object)serialNumber);
        KeyStoreContentFetcher fetcher = new KeyStoreContentFetcher(keyStore);
        fetcher.keyPassword(password);
        Optional<PrivateKey> privateKey = fetcher.getPrivateKeyFor(c -> cert.equals(c));
        fetcher.destroy();
        if (privateKey.isPresent()) {
            log.trace("Found private key: {}", privateKey);
            return privateKey.get();
        }
        log.debug("No private key found for certificate with serial number {} in key store.", (Object)serialNumber);
        return null;
    }

    public static final String[] getEmails(X509Certificate certificate) throws CertificateEncodingException, CertificateParsingException {
        if (certificate == null) {
            return new String[0];
        }
        String[] emails = SmimeUtil.getEmails(SmimeUtil.getSubject(certificate), certificate.getSubjectAlternativeNames());
        log.trace("Emails of subject from certificate with serial number {}: {}", (Object)SmimeUtil.getSerialNumber(certificate), (Object)emails);
        return emails;
    }

    public static final String[] getIssuerEmails(X509Certificate certificate) throws CertificateEncodingException, CertificateParsingException {
        if (certificate == null) {
            return new String[0];
        }
        String[] emails = SmimeUtil.getEmails(SmimeUtil.getIssuer(certificate), certificate.getIssuerAlternativeNames());
        log.trace("Emails of issuer from certificate with serial number {}: {}", (Object)SmimeUtil.getSerialNumber(certificate), (Object)emails);
        return emails;
    }

    private static final String[] getEmails(X500Name name, Collection<List<?>> altNames) {
        String email;
        LinkedHashSet<String> emails = new LinkedHashSet<String>();
        if (name != null && (email = SmimeUtil.getRDNValue(name, BCStyle.E)) != null) {
            emails.add(email);
        }
        if (altNames != null) {
            for (List<?> item : altNames) {
                int type = (Integer)item.get(0);
                Object value = item.get(1);
                if (type != 1 || value == null) continue;
                emails.add(value.toString());
            }
        }
        return emails.toArray(new String[emails.size()]);
    }

    public static final boolean isCertForPurpose(X509Certificate cert, EKeyPurpose purpose) {
        boolean result;
        if (cert == null || purpose == null) {
            return false;
        }
        log.debug("Checking purpose {} for certificate with serial number: {}", (Object)purpose, (Object)SmimeUtil.getSerialNumber(cert));
        Set<String> criticalExtensions = cert.getCriticalExtensionOIDs();
        if (criticalExtensions != null && criticalExtensions.contains("2.5.29.37")) {
            try {
                List<String> extendedKeyUsage = cert.getExtendedKeyUsage();
                if (extendedKeyUsage == null || !extendedKeyUsage.contains("1.3.6.1.5.5.7.3.4")) {
                    log.debug("Extended Key Usage for certificate is critical, but does not contain 'Email'.");
                    return false;
                }
            }
            catch (CertificateParsingException e) {
                log.error("Could not parse 'Extended Key Usage' extension of certificate " + SmimeUtil.getSerialNumber(cert) + ". Error message: " + e.getMessage(), (Throwable)e);
                return false;
            }
        }
        if (criticalExtensions == null || !criticalExtensions.contains("2.5.29.15")) {
            log.debug("Key usage is not critical.");
            return true;
        }
        boolean[] usage = cert.getKeyUsage();
        switch (purpose) {
            case DECRYPTION: 
            case ENCRYPTION: {
                result = usage[2];
                break;
            }
            case SIGNING: {
                result = usage[0];
                break;
            }
            case UNDEFINED: {
                result = false;
                break;
            }
            default: {
                throw new IllegalStateException("Handling of key purpose " + (Object)((Object)purpose) + " not yet implemented.");
            }
        }
        log.debug("Certificate is usable: {}", (Object)result);
        return result;
    }

    public static final boolean isCertForEmail(X509Certificate certificate, String email) throws CertificateEncodingException, CertificateParsingException {
        if (email == null) {
            return false;
        }
        for (String foundEmail : SmimeUtil.getEmails(certificate)) {
            if (!email.equalsIgnoreCase(foundEmail)) continue;
            return true;
        }
        return false;
    }

    public static final X500Name getIssuer(X509Certificate certificate) {
        return new X500Name(certificate.getIssuerX500Principal().getName());
    }

    public static final X500Name getSubject(X509Certificate certificate) {
        return new X500Name(certificate.getSubjectX500Principal().getName());
    }

    public static final String getRDNValue(X500Name name, ASN1ObjectIdentifier identifier) {
        RDN[] rdns = name.getRDNs(identifier);
        if (rdns.length == 0) {
            return null;
        }
        return rdns[0].getFirst().getValue().toString();
    }

    public static final KeyStore convertToKeyStore(Store<X509CertificateHolder> store, String type, char[] password) throws KeyStoreException, CertificateException {
        KeyStore convertedKeyStore = KeyStore.getInstance(type, SecurityUtil.getProvider());
        try {
            convertedKeyStore.load(null, password);
        }
        catch (IOException | NoSuchAlgorithmException e) {
            log.error("Failed to create key store. Error message: " + e.getMessage(), (Throwable)e);
        }
        JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider(SecurityUtil.getProvider());
        Collection<X509CertificateHolder> allHolders = store.getMatches(null);
        log.debug("Converting {} certificates.", (Object)allHolders.size());
        for (X509CertificateHolder holder : allHolders) {
            String alias = holder.getIssuer().toString() + holder.getSerialNumber().toString();
            log.trace("Adding certificate with alias \"{}\" to key store.", (Object)alias);
            X509Certificate cert = converter.getCertificate(holder);
            convertedKeyStore.setCertificateEntry(alias, cert);
        }
        log.debug("Added all certificates to key store.");
        return convertedKeyStore;
    }

    public static final String getCmsName(ASN1ObjectIdentifier algorithm) {
        if (algorithm == null) {
            return "null";
        }
        if (algorithm.equals(CMSAlgorithm.AES128_CBC)) {
            return "AES128-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES192_CBC)) {
            return "AES192-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES256_CBC)) {
            return "AES256-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES128_GCM)) {
            return "AES128-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.AES192_GCM)) {
            return "AES192-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.AES256_GCM)) {
            return "AES256-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.DES_CBC)) {
            return "DES-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.DES_EDE3_CBC)) {
            return "DES-EDE3-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.CAST5_CBC)) {
            return "CAST5-CBC";
        }
        if (algorithm.equals(PKCSObjectIdentifiers.rsaEncryption)) {
            return "RSA";
        }
        return "Unknown (" + algorithm.getId() + ")";
    }
}

