/*
 * Decompiled with CFR 0.152.
 */
package net.savignano.cryptography.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import net.savignano.cryptography.LazyLog;
import net.savignano.cryptography.enums.EKeyPurpose;
import net.savignano.cryptography.util.KeyStoreContentFetcher;
import net.savignano.cryptography.util.SecurityUtil;
import net.savignano.thirdparty.org.bouncycastle.asn1.ASN1ObjectIdentifier;
import net.savignano.thirdparty.org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.RDN;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.X500Name;
import net.savignano.thirdparty.org.bouncycastle.asn1.x500.style.BCStyle;
import net.savignano.thirdparty.org.bouncycastle.cert.X509CertificateHolder;
import net.savignano.thirdparty.org.bouncycastle.cert.jcajce.JcaCertStore;
import net.savignano.thirdparty.org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSAbsentContent;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSAlgorithm;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSException;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSSignedData;
import net.savignano.thirdparty.org.bouncycastle.cms.CMSSignedDataGenerator;
import net.savignano.thirdparty.org.bouncycastle.gpg.keybox.KeyBox;
import net.savignano.thirdparty.org.bouncycastle.gpg.keybox.bc.BcKeyBox;
import net.savignano.thirdparty.org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SmimeUtil {
    public static final String X509_FACTORY_KEY = "X.509";
    private static final Logger log = LoggerFactory.getLogger(SmimeUtil.class);

    public static final Predicate<X509Certificate> createPredicateForEmail(String email) {
        return c -> {
            try {
                boolean isForEmail = Arrays.asList(SmimeUtil.getEmails(c)).contains(email);
                if (log.isTraceEnabled()) {
                    log.trace("Certificate for \"{}\" (Serial Number: {}) is for email <{}>: {}", new Object[]{SmimeUtil.getSubject(c), SmimeUtil.getSerialNumber(c), email, isForEmail});
                }
                return isForEmail;
            }
            catch (CertificateEncodingException | CertificateParsingException e) {
                log.error(e.getMessage(), (Throwable)e);
                return false;
            }
        };
    }

    public static final Predicate<X509Certificate> createPredicateForPurpose(EKeyPurpose purpose) {
        return c -> SmimeUtil.isCertForPurpose(c, purpose);
    }

    public static final Comparator<X509Certificate> getCertificateComparator() {
        return Comparator.nullsLast((c1, c2) -> {
            boolean valid2;
            Date now = new Date();
            Date since1 = c1.getNotBefore();
            Date since2 = c2.getNotBefore();
            Date until1 = c1.getNotAfter();
            Date until2 = c2.getNotAfter();
            boolean valid1 = since1.before(now) && until1.after(now);
            boolean bl = valid2 = since2.before(now) && until2.after(now);
            if (valid1 == valid2) {
                int newer = since2.compareTo(since1);
                if (newer != 0) {
                    return newer;
                }
                return until2.compareTo(until1);
            }
            if (valid1) {
                return -1;
            }
            if (valid2) {
                return 1;
            }
            return 0;
        });
    }

    public static final String getSerialNumber(X509Certificate certificate) {
        if (certificate == null) {
            log.debug("Null value given for certificate.");
            return "<N/A>";
        }
        BigInteger serialNumber = certificate.getSerialNumber();
        if (serialNumber == null) {
            log.warn("No serial number in certificate. Might be corrupt. Certificate: {}", (Object)certificate);
            return "<N/A>";
        }
        return SmimeUtil.getSerialNumber(serialNumber);
    }

    public static final String getSerialNumber(BigInteger serialNumber) {
        if (serialNumber == null) {
            log.debug("Null value given for serial number.");
            return "<N/A>";
        }
        return serialNumber.toString() + " (0x" + serialNumber.toString(16) + ")";
    }

    public static final boolean isSupportedSymmetricKeyAlgorithm(ASN1ObjectIdentifier algorithm) {
        if (algorithm == null) {
            return false;
        }
        return !algorithm.equals(CMSAlgorithm.DES_CBC);
    }

    public static final boolean isSupportedAsymmetricKeyAlgorithm(ASN1ObjectIdentifier algorithm) {
        return algorithm != null;
    }

    public static final KeyBox loadKeyBox(InputStream is) throws IOException {
        return new BcKeyBox(new BufferedInputStream(is));
    }

    public static final X509Certificate createCertificate(byte[] content) throws CertificateException {
        ByteArrayInputStream is = new ByteArrayInputStream(content);
        CertificateFactory fact = CertificateFactory.getInstance(X509_FACTORY_KEY);
        return (X509Certificate)fact.generateCertificate(is);
    }

    public static final PrivateKey getPrivateKeyForCert(KeyStore keyStore, X509Certificate cert, char[] password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException {
        if (keyStore == null || cert == null) {
            return null;
        }
        log.debug("Getting private key for certificate with serial number: {}", (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(cert)));
        KeyStoreContentFetcher fetcher = new KeyStoreContentFetcher(keyStore);
        fetcher.keyPassword(password);
        Optional<PrivateKey> privateKey = fetcher.getPrivateKeyFor(c -> cert.equals(c));
        fetcher.destroy();
        if (privateKey.isPresent()) {
            log.trace("Found private key: {}", privateKey);
            return privateKey.get();
        }
        log.debug("No private key found.");
        return null;
    }

    public static final String[] getEmails(X509Certificate certificate) throws CertificateEncodingException, CertificateParsingException {
        if (certificate == null) {
            return new String[0];
        }
        String[] emails = SmimeUtil.getEmails(SmimeUtil.getSubject(certificate), certificate.getSubjectAlternativeNames());
        log.trace("Emails of subject from certificate with serial number {}: {}", (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(certificate)), (Object)emails);
        return emails;
    }

    public static final String[] getIssuerEmails(X509Certificate certificate) throws CertificateEncodingException, CertificateParsingException {
        if (certificate == null) {
            return new String[0];
        }
        String[] emails = SmimeUtil.getEmails(SmimeUtil.getIssuer(certificate), certificate.getIssuerAlternativeNames());
        log.trace("Emails of issuer from certificate with serial number {}: {}", (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(certificate)), (Object)emails);
        return emails;
    }

    private static final String[] getEmails(X500Name name, Collection<List<?>> altNames) {
        String email;
        LinkedHashSet<String> emails = new LinkedHashSet<String>();
        if (name != null && (email = SmimeUtil.getRDNValue(name, BCStyle.E)) != null) {
            emails.add(email);
        }
        if (altNames != null) {
            for (List<?> item : altNames) {
                int type = (Integer)item.get(0);
                Object value = item.get(1);
                if (type != 1 || value == null) continue;
                emails.add(value.toString());
            }
        }
        return emails.toArray(new String[emails.size()]);
    }

    public static final boolean isCertCurrentlyValid(X509Certificate cert) {
        if (cert == null) {
            return false;
        }
        boolean valid = true;
        try {
            cert.checkValidity();
        }
        catch (CertificateExpiredException | CertificateNotYetValidException e) {
            valid = false;
        }
        if (log.isTraceEnabled()) {
            log.trace("Certificate for \"{}\" (Serial Number: {}) is currently ({}) valid: {}", new Object[]{SmimeUtil.getSubject(cert), SmimeUtil.getSerialNumber(cert), DateFormat.getInstance().format(new Date()), valid});
        }
        return valid;
    }

    public static final boolean isCertForPurpose(X509Certificate cert, EKeyPurpose purpose) {
        boolean result;
        if (cert == null || purpose == null) {
            return false;
        }
        log.debug("Checking purpose {} for certificate with serial number: {}", (Object)purpose, (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(cert)));
        Set<String> criticalExtensions = cert.getCriticalExtensionOIDs();
        if (criticalExtensions != null && criticalExtensions.contains("2.5.29.37")) {
            try {
                List<String> extendedKeyUsage = cert.getExtendedKeyUsage();
                if (extendedKeyUsage == null || !extendedKeyUsage.contains("1.3.6.1.5.5.7.3.4")) {
                    log.debug("Extended Key Usage for certificate is critical, but does not contain 'Email'.");
                    return false;
                }
            }
            catch (CertificateParsingException e) {
                log.error("Could not parse 'Extended Key Usage' extension of certificate " + SmimeUtil.getSerialNumber(cert) + ". Error message: " + e.getMessage(), (Throwable)e);
                return false;
            }
        }
        if (criticalExtensions == null || !criticalExtensions.contains("2.5.29.15")) {
            log.debug("Key usage is not critical.");
            return true;
        }
        boolean[] usage = cert.getKeyUsage();
        switch (purpose) {
            case DECRYPTION: 
            case ENCRYPTION: {
                result = usage[2];
                break;
            }
            case SIGNING: 
            case VALIDATION: {
                result = usage[0];
                break;
            }
            case UNDEFINED: {
                result = false;
                break;
            }
            default: {
                throw new IllegalStateException("Handling of key purpose " + (Object)((Object)purpose) + " not yet implemented.");
            }
        }
        log.debug("Certificate is usable: {}", (Object)result);
        return result;
    }

    public static final boolean isCertCriticalExtensionSatisfied(X509Certificate cert) {
        if (cert == null) {
            return false;
        }
        Set<String> criticalExtensions = cert.getCriticalExtensionOIDs();
        if (criticalExtensions == null) {
            log.debug("Certificate with serial number {} has no critical extensions defined.", (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(cert)));
            return true;
        }
        List<String> supportedExtensions = Arrays.asList("2.5.29.37", "2.5.29.15", "2.5.29.17", "2.5.29.31", "1.3.6.1.5.5.7.48.1");
        if (supportedExtensions.containsAll(criticalExtensions)) {
            log.debug("Certificate with serial number {} only has supported critical extensions.", (Object)new LazyLog(() -> SmimeUtil.getSerialNumber(cert)));
            return true;
        }
        ArrayList<String> unsupportedExtensions = new ArrayList<String>(criticalExtensions);
        unsupportedExtensions.removeAll(supportedExtensions);
        log.warn("Certificate with serial number {} cannot be used, as it contains critical extensions that are not supported by S/Notify. Unsupported extension(s): {}", (Object)SmimeUtil.getSerialNumber(cert), unsupportedExtensions);
        return false;
    }

    public static final boolean isCertForEmail(X509Certificate certificate, String email) throws CertificateEncodingException, CertificateParsingException {
        if (email == null) {
            return false;
        }
        for (String foundEmail : SmimeUtil.getEmails(certificate)) {
            if (!email.equalsIgnoreCase(foundEmail)) continue;
            return true;
        }
        return false;
    }

    public static final X500Name getIssuer(X509Certificate certificate) {
        return new X500Name(certificate.getIssuerX500Principal().getName());
    }

    public static final X500Name getSubject(X509Certificate certificate) {
        return new X500Name(certificate.getSubjectX500Principal().getName());
    }

    public static final String getRDNValue(X500Name name, ASN1ObjectIdentifier identifier) {
        RDN[] rdns = name.getRDNs(identifier);
        if (rdns.length == 0) {
            return null;
        }
        return rdns[0].getFirst().getValue().toString();
    }

    public static final KeyStore convertToKeyStore(Store<X509CertificateHolder> store, String type, char[] password) throws KeyStoreException, CertificateException {
        KeyStore convertedKeyStore = KeyStore.getInstance(type, SecurityUtil.getProvider());
        try {
            convertedKeyStore.load(null, password);
        }
        catch (IOException | NoSuchAlgorithmException e) {
            log.error("Failed to create key store. Error message: " + e.getMessage(), (Throwable)e);
        }
        JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider(SecurityUtil.getProvider());
        Collection<X509CertificateHolder> allHolders = store.getMatches(null);
        log.debug("Converting {} certificates.", (Object)allHolders.size());
        for (X509CertificateHolder holder : allHolders) {
            String alias = holder.getIssuer().toString() + holder.getSerialNumber().toString();
            log.trace("Adding certificate with alias \"{}\" to key store.", (Object)alias);
            X509Certificate cert = converter.getCertificate(holder);
            convertedKeyStore.setCertificateEntry(alias, cert);
        }
        log.debug("Added all certificates to key store.");
        return convertedKeyStore;
    }

    public static final CMSSignedData convertToP7b(KeyStore keyStore) throws CertificateEncodingException, KeyStoreException, CMSException {
        KeyStoreContentFetcher fetcher = new KeyStoreContentFetcher(keyStore);
        Set<X509Certificate> certs = fetcher.getCertsFor(c -> true);
        log.debug("Converting {} certificates.", (Object)certs.size());
        JcaCertStore certStore = new JcaCertStore(certs);
        CMSSignedDataGenerator generator = new CMSSignedDataGenerator();
        generator.addCertificates(certStore);
        CMSSignedData p7b = generator.generate(new CMSAbsentContent());
        log.debug("Added all certificates to P7B.");
        return p7b;
    }

    public static final String getCmsName(ASN1ObjectIdentifier algorithm) {
        if (algorithm == null) {
            return "null";
        }
        if (algorithm.equals(CMSAlgorithm.AES128_CBC)) {
            return "AES128-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES192_CBC)) {
            return "AES192-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES256_CBC)) {
            return "AES256-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.AES128_GCM)) {
            return "AES128-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.AES192_GCM)) {
            return "AES192-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.AES256_GCM)) {
            return "AES256-GCM";
        }
        if (algorithm.equals(CMSAlgorithm.DES_CBC)) {
            return "DES-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.DES_EDE3_CBC)) {
            return "DES-EDE3-CBC";
        }
        if (algorithm.equals(CMSAlgorithm.CAST5_CBC)) {
            return "CAST5-CBC";
        }
        if (algorithm.equals(PKCSObjectIdentifiers.rsaEncryption)) {
            return "RSA";
        }
        return "Unknown (" + algorithm.getId() + ")";
    }
}

