/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.cli;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Stopwatch;
import com.google.common.base.Strings;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.james.cli.exceptions.InvalidArgumentNumberException;
import org.apache.james.cli.exceptions.JamesCliException;
import org.apache.james.cli.exceptions.MissingCommandException;
import org.apache.james.cli.exceptions.UnrecognizedCommandException;
import org.apache.james.cli.probe.impl.JmxConnection;
import org.apache.james.cli.probe.impl.JmxDataProbe;
import org.apache.james.cli.probe.impl.JmxMailboxProbe;
import org.apache.james.cli.probe.impl.JmxQuotaProbe;
import org.apache.james.cli.probe.impl.JmxSieveProbe;
import org.apache.james.cli.type.CmdType;
import org.apache.james.core.quota.QuotaCountLimit;
import org.apache.james.core.quota.QuotaCountUsage;
import org.apache.james.core.quota.QuotaLimitValue;
import org.apache.james.core.quota.QuotaSizeLimit;
import org.apache.james.core.quota.QuotaSizeUsage;
import org.apache.james.mailbox.model.SerializableQuota;
import org.apache.james.mailbox.model.SerializableQuotaLimitValue;
import org.apache.james.rrt.lib.Mappings;
import org.apache.james.util.Port;
import org.apache.james.util.Size;
import org.apache.james.util.SizeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerCmd {
    public static final String HOST_OPT_LONG = "host";
    public static final String HOST_OPT_SHORT = "h";
    public static final String PORT_OPT_LONG = "port";
    public static final String PORT_OPT_SHORT = "p";
    private static final String DEFAULT_HOST = "127.0.0.1";
    private static final int DEFAULT_PORT = 9999;
    private static final Logger LOG = LoggerFactory.getLogger(ServerCmd.class);
    private final JmxDataProbe probe;
    private final JmxMailboxProbe mailboxProbe;
    private final JmxQuotaProbe quotaProbe;
    private final JmxSieveProbe sieveProbe;

    private static Options createOptions() {
        return new Options().addOption(HOST_OPT_SHORT, HOST_OPT_LONG, true, "node hostname or ip address").addOption(PORT_OPT_SHORT, PORT_OPT_LONG, true, "remote jmx agent port number");
    }

    public static void main(String[] args) {
        try {
            ServerCmd.doMain(args);
            System.exit(0);
        }
        catch (JamesCliException e) {
            ServerCmd.failWithMessage(e.getMessage());
        }
        catch (ParseException e) {
            ServerCmd.failWithMessage("Error parsing command line : " + e.getMessage());
        }
        catch (IOException ioe) {
            ServerCmd.failWithMessage("Error connecting to remote JMX agent : " + ioe.getMessage());
        }
        catch (Exception e) {
            LOG.error("Error while playing command", (Throwable)e);
            ServerCmd.failWithMessage("Error " + String.valueOf(e.getClass()) + " while executing command:" + e.getMessage());
        }
    }

    public static void doMain(String[] args) throws Exception {
        PrintStream printStream = System.out;
        ServerCmd.executeAndOutputToStream(args, printStream);
    }

    public static void executeAndOutputToStream(String[] args, PrintStream printStream) throws Exception {
        Stopwatch stopWatch = Stopwatch.createStarted();
        CommandLine cmd = ServerCmd.parseCommandLine(args);
        JmxConnection jmxConnection = new JmxConnection(ServerCmd.getHost(cmd), ServerCmd.getPort(cmd));
        CmdType cmdType = new ServerCmd(new JmxDataProbe().connect(jmxConnection), new JmxMailboxProbe().connect(jmxConnection), new JmxQuotaProbe().connect(jmxConnection), new JmxSieveProbe().connect(jmxConnection)).executeCommandLine(cmd, printStream);
        ServerCmd.print(new String[]{Joiner.on((char)' ').join((Object)cmdType.getCommand(), (Object)"command executed sucessfully in", new Object[]{stopWatch.elapsed(TimeUnit.MILLISECONDS), "ms."})}, printStream);
        stopWatch.stop();
    }

    public ServerCmd(JmxDataProbe probe, JmxMailboxProbe mailboxProbe, JmxQuotaProbe quotaProbe, JmxSieveProbe sieveProbe) {
        this.probe = probe;
        this.mailboxProbe = mailboxProbe;
        this.quotaProbe = quotaProbe;
        this.sieveProbe = sieveProbe;
    }

    @VisibleForTesting
    static CommandLine parseCommandLine(String[] args) throws ParseException {
        DefaultParser parser = new DefaultParser();
        CommandLine commandLine = parser.parse(ServerCmd.createOptions(), args);
        if (commandLine.getArgs().length < 1) {
            throw new MissingCommandException();
        }
        return commandLine;
    }

    @VisibleForTesting
    static String getHost(CommandLine cmd) {
        String host = cmd.getOptionValue(HOST_OPT_LONG);
        if (Strings.isNullOrEmpty((String)host)) {
            return DEFAULT_HOST;
        }
        return host;
    }

    @VisibleForTesting
    static int getPort(CommandLine cmd) throws ParseException {
        String portNum = cmd.getOptionValue(PORT_OPT_LONG);
        if (!Strings.isNullOrEmpty((String)portNum)) {
            try {
                int portNumber = Integer.parseInt(portNum);
                Port.assertValid((int)portNumber);
                return portNumber;
            }
            catch (NumberFormatException e) {
                throw new ParseException("Port must be a number");
            }
        }
        return 9999;
    }

    private static void failWithMessage(String s) {
        System.err.println(s);
        ServerCmd.printUsage();
        System.exit(1);
    }

    @VisibleForTesting
    private CmdType executeCommandLine(CommandLine commandLine, PrintStream printStream) throws Exception {
        String[] arguments = commandLine.getArgs();
        String cmdName = arguments[0];
        CmdType cmdType = CmdType.lookup(cmdName);
        if (cmdType == null) {
            throw new UnrecognizedCommandException(cmdName);
        }
        if (!cmdType.hasCorrectArguments(arguments.length)) {
            throw new InvalidArgumentNumberException(cmdType, arguments.length);
        }
        this.executeCommand(arguments, cmdType, printStream);
        return cmdType;
    }

    @VisibleForTesting
    CmdType executeCommandLine(CommandLine commandLine) throws Exception {
        return this.executeCommandLine(commandLine, new PrintStream(System.out));
    }

    private void executeCommand(String[] arguments, CmdType cmdType, PrintStream printStream) throws Exception {
        switch (cmdType) {
            case ADDUSER: {
                this.probe.addUser(arguments[1], arguments[2]);
                break;
            }
            case REMOVEUSER: {
                this.probe.removeUser(arguments[1]);
                break;
            }
            case LISTUSERS: {
                ServerCmd.print(this.probe.listUsers(), printStream);
                break;
            }
            case ADDDOMAIN: {
                this.probe.addDomain(arguments[1]);
                break;
            }
            case REMOVEDOMAIN: {
                this.probe.removeDomain(arguments[1]);
                break;
            }
            case CONTAINSDOMAIN: {
                if (this.probe.containsDomain(arguments[1])) {
                    printStream.println(arguments[1] + " exists");
                    break;
                }
                printStream.println(arguments[1] + " does not exists");
                break;
            }
            case LISTDOMAINS: {
                ServerCmd.print(this.probe.listDomains(), printStream);
                break;
            }
            case LISTMAPPINGS: {
                this.print(this.probe.listMappings(), printStream);
                break;
            }
            case LISTUSERDOMAINMAPPINGS: {
                Mappings userDomainMappings = this.probe.listUserDomainMappings(arguments[1], arguments[2]);
                ServerCmd.print(userDomainMappings.asStrings(), printStream);
                break;
            }
            case ADDADDRESSMAPPING: {
                this.probe.addAddressMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REMOVEADDRESSMAPPING: {
                this.probe.removeAddressMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case ADDREGEXMAPPING: {
                this.probe.addRegexMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REMOVEREGEXMAPPING: {
                this.probe.removeRegexMapping(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case SETPASSWORD: {
                this.probe.setPassword(arguments[1], arguments[2]);
                break;
            }
            case COPYMAILBOX: {
                this.mailboxProbe.copyMailbox(arguments[1], arguments[2]);
                break;
            }
            case DELETEUSERMAILBOXES: {
                this.mailboxProbe.deleteUserMailboxesNames(arguments[1]);
                break;
            }
            case CREATEMAILBOX: {
                this.mailboxProbe.createMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case LISTUSERMAILBOXES: {
                Collection<String> mailboxes = this.mailboxProbe.listUserMailboxes(arguments[1]);
                ServerCmd.print((String[])mailboxes.toArray(String[]::new), printStream);
                break;
            }
            case DELETEMAILBOX: {
                this.mailboxProbe.deleteMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case IMPORTEML: {
                this.mailboxProbe.importEmlFileToMailbox(arguments[1], arguments[2], arguments[3], arguments[4]);
                break;
            }
            case GETSTORAGEQUOTA: {
                this.printStorageQuota(arguments[1], this.quotaProbe.getStorageQuota(arguments[1]), printStream);
                break;
            }
            case GETMESSAGECOUNTQUOTA: {
                this.printMessageQuota(arguments[1], this.quotaProbe.getMessageCountQuota(arguments[1]), printStream);
                break;
            }
            case GETQUOTAROOT: {
                printStream.println("Quota Root: " + this.quotaProbe.getQuotaRoot(arguments[1], arguments[2], arguments[3]));
                break;
            }
            case GETMAXSTORAGEQUOTA: {
                printStream.println("Storage space allowed for Quota Root " + arguments[1] + ": " + this.formatStorageValue(this.quotaProbe.getMaxStorage(arguments[1])));
                break;
            }
            case GETMAXMESSAGECOUNTQUOTA: {
                printStream.println("MailboxMessage count allowed for Quota Root " + arguments[1] + ": " + this.formatMessageValue(this.quotaProbe.getMaxMessageCount(arguments[1])));
                break;
            }
            case SETMAXSTORAGEQUOTA: {
                this.quotaProbe.setMaxStorage(arguments[1], this.parseQuotaSize(arguments[2]));
                break;
            }
            case SETMAXMESSAGECOUNTQUOTA: {
                this.quotaProbe.setMaxMessageCount(arguments[1], this.parseQuotaCount(arguments[2]));
                break;
            }
            case SETGLOBALMAXSTORAGEQUOTA: {
                this.quotaProbe.setGlobalMaxStorage(this.parseQuotaSize(arguments[1]));
                break;
            }
            case SETGLOBALMAXMESSAGECOUNTQUOTA: {
                this.quotaProbe.setGlobalMaxMessageCount(this.parseQuotaCount(arguments[1]));
                break;
            }
            case GETGLOBALMAXSTORAGEQUOTA: {
                printStream.println("Global Maximum Storage Quota: " + this.formatStorageValue(this.quotaProbe.getGlobalMaxStorage()));
                break;
            }
            case GETGLOBALMAXMESSAGECOUNTQUOTA: {
                printStream.println("Global Maximum message count Quota: " + this.formatMessageValue(this.quotaProbe.getGlobalMaxMessageCount()));
                break;
            }
            case REINDEXMAILBOX: {
                this.mailboxProbe.reIndexMailbox(arguments[1], arguments[2], arguments[3]);
                break;
            }
            case REINDEXALL: {
                this.mailboxProbe.reIndexAll();
                break;
            }
            case SETSIEVEQUOTA: {
                this.sieveProbe.setSieveQuota(Size.parse((String)arguments[1]).asBytes());
                break;
            }
            case SETSIEVEUSERQUOTA: {
                this.sieveProbe.setSieveQuota(arguments[1], Size.parse((String)arguments[2]).asBytes());
                break;
            }
            case GETSIEVEQUOTA: {
                printStream.println("Storage space allowed for Sieve scripts by default: " + this.formatStorageValue(this.sieveProbe.getSieveQuota()));
                break;
            }
            case GETSIEVEUSERQUOTA: {
                printStream.println("Storage space allowed for " + arguments[1] + " Sieve scripts: " + this.formatStorageValue(this.sieveProbe.getSieveQuota(arguments[1])));
                break;
            }
            case REMOVESIEVEQUOTA: {
                this.sieveProbe.removeSieveQuota();
                break;
            }
            case REMOVESIEVEUSERQUOTA: {
                this.sieveProbe.removeSieveQuota(arguments[1]);
                break;
            }
            case ADDACTIVESIEVESCRIPT: {
                this.sieveProbe.addActiveSieveScriptFromFile(arguments[1], arguments[2], arguments[3]);
                break;
            }
            default: {
                throw new UnrecognizedCommandException(cmdType.getCommand());
            }
        }
    }

    private SerializableQuotaLimitValue<QuotaSizeLimit> parseQuotaSize(String argument) throws Exception {
        long convertedValue = Size.parse((String)argument).asBytes();
        return this.longToSerializableQuotaValue(convertedValue, QuotaSizeLimit.unlimited(), QuotaSizeLimit::size);
    }

    private SerializableQuotaLimitValue<QuotaCountLimit> parseQuotaCount(String argument) {
        long value = Long.parseLong(argument);
        return this.longToSerializableQuotaValue(value, QuotaCountLimit.unlimited(), QuotaCountLimit::count);
    }

    private <T extends QuotaLimitValue<T>> SerializableQuotaLimitValue<T> longToSerializableQuotaValue(long value, T unlimited, Function<Long, T> factory) {
        return SerializableQuotaLimitValue.valueOf(Optional.of(this.longToQuotaValue(value, unlimited, factory)));
    }

    private <T extends QuotaLimitValue<T>> T longToQuotaValue(long value, T unlimited, Function<Long, T> factory) {
        if (value == -1L) {
            return unlimited;
        }
        if (value >= 0L) {
            return (T)((QuotaLimitValue)factory.apply(value));
        }
        throw new IllegalArgumentException("Quota should be -1 for unlimited or a positive value");
    }

    private static void print(String[] data, PrintStream out) {
        ServerCmd.print(Arrays.asList(data), out);
    }

    private static void print(Iterable<String> data, PrintStream out) {
        if (data != null) {
            out.println(Joiner.on((char)'\n').join(data));
        }
    }

    private static void printUsage() {
        StringBuilder footerBuilder = new StringBuilder();
        for (CmdType cmdType : CmdType.values()) {
            footerBuilder.append(cmdType.getUsage()).append("\n");
        }
        new HelpFormatter().printHelp(String.format("java %s --host <arg> <command>%n", ServerCmd.class.getName()), "", ServerCmd.createOptions(), footerBuilder.toString());
    }

    private void printStorageQuota(String quotaRootString, SerializableQuota<QuotaSizeLimit, QuotaSizeUsage> quota, PrintStream printStream) {
        printStream.println(String.format("Storage quota for %s is: %s / %s", quotaRootString, this.formatStorageValue(quota.getUsed()), this.formatStorageValue(quota.encodeAsLong())));
    }

    private void printMessageQuota(String quotaRootString, SerializableQuota<QuotaCountLimit, QuotaCountUsage> quota, PrintStream printStream) {
        printStream.println(String.format("MailboxMessage count quota for %s is: %s / %s", quotaRootString, this.formatMessageValue(quota.getUsed()), this.formatMessageValue(quota.encodeAsLong())));
    }

    private String formatStorageValue(Long value) {
        if (value == null) {
            return "UNKNOWN";
        }
        if (value == -1L) {
            return "UNLIMITED";
        }
        return SizeFormat.format((long)value);
    }

    private String formatStorageValue(SerializableQuotaLimitValue<QuotaSizeLimit> value) {
        return value.toValue(QuotaSizeLimit::size, (QuotaLimitValue)QuotaSizeLimit.unlimited()).map(size -> {
            if (size.isUnlimited()) {
                return "UNLIMITED";
            }
            return SizeFormat.format((long)size.asLong());
        }).orElse("UNKNOWN");
    }

    private String formatMessageValue(Long value) {
        if (value == null) {
            return "UNKNOWN";
        }
        if (value == -1L) {
            return "UNLIMITED";
        }
        return String.valueOf(value);
    }

    private String formatMessageValue(SerializableQuotaLimitValue<QuotaCountLimit> value) {
        return value.toValue(QuotaCountLimit::count, (QuotaLimitValue)QuotaCountLimit.unlimited()).map(count -> {
            if (count.isUnlimited()) {
                return "UNLIMITED";
            }
            return String.valueOf(count.asLong());
        }).orElse("UNKNOWN");
    }

    private void print(Map<String, Mappings> map, PrintStream out) {
        if (map != null) {
            for (Map.Entry<String, Mappings> entry : map.entrySet()) {
                out.println(entry.getKey() + "=" + entry.getValue().serialize());
            }
            out.println();
        }
    }
}

