/*
 * Decompiled with CFR 0.152.
 */
package net.savignano.uptrust.config.features.auth.service;

import io.github.bucket4j.Bandwidth;
import io.github.bucket4j.Bucket;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseEntity;
import org.springframework.security.crypto.bcrypt.BCrypt;
import org.springframework.stereotype.Service;

@Service
public class AuthService {
    private static final Logger logger = LoggerFactory.getLogger(AuthService.class);
    private static final String PASSWORD_FILE = "pwd";
    private static final String DEFAULT_USERNAME = "admin";
    private static final String DEFAULT_PASS = "uptrust";
    private static final String HASH_ALGORITHM = "bcrypt";
    private final Map<String, Bucket> buckets = new ConcurrentHashMap();

    public AuthService() throws IOException {
        this.ensurePasswordFileExists();
    }

    public ResponseEntity<String> authenticate(String username, String password, String clientIP) {
        Bucket bucket = this.getBucket(clientIP);
        if (!bucket.tryConsume(1L)) {
            logger.warn("Rate limit exceeded for IP: {}", (Object)clientIP);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.TOO_MANY_REQUESTS).body((Object)"Too many requests - please try again later");
        }
        try {
            String[] credentials = this.loadCredentials();
            String storedHash = credentials[2];
            if (!DEFAULT_USERNAME.equals(username)) {
                logger.warn("Login attempt with invalid username: {}", (Object)username);
                return ResponseEntity.status((HttpStatusCode)HttpStatus.UNAUTHORIZED).body((Object)"Invalid credentials");
            }
            if (!BCrypt.checkpw((String)password, (String)storedHash)) {
                logger.warn("Failed login attempt for user: {}", (Object)username);
                return ResponseEntity.status((HttpStatusCode)HttpStatus.UNAUTHORIZED).body((Object)"Invalid credentials");
            }
            boolean defaultPass = BCrypt.checkpw((String)DEFAULT_PASS, (String)storedHash);
            logger.info("User {} logged in successfully", (Object)username);
            if (defaultPass) {
                return ((ResponseEntity.BodyBuilder)ResponseEntity.status((HttpStatusCode)HttpStatus.OK).header("x-password-change", new String[]{"true"})).body((Object)"Password change required");
            }
            return ResponseEntity.ok((Object)"Login successful");
        }
        catch (IOException e) {
            logger.error("Login process failed", (Throwable)e);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.INTERNAL_SERVER_ERROR).body((Object)"System error occurred");
        }
    }

    public ResponseEntity<String> changePassword(String oldPassword, String newPassword, String clientIP) {
        Bucket bucket = this.getBucket(clientIP);
        if (!bucket.tryConsume(1L)) {
            logger.warn("Rate limit exceeded for password change from IP: {}", (Object)clientIP);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.TOO_MANY_REQUESTS).body((Object)"Too many requests - please try again later");
        }
        try {
            String[] credentials = this.loadCredentials();
            String storedHash = credentials[2];
            if (!BCrypt.checkpw((String)oldPassword, (String)storedHash)) {
                logger.warn("Invalid old password during change attempt");
                return ResponseEntity.status((HttpStatusCode)HttpStatus.UNAUTHORIZED).body((Object)"Invalid old password");
            }
            String newHash = BCrypt.hashpw((String)newPassword, (String)BCrypt.gensalt());
            Files.write(Path.of(PASSWORD_FILE, new String[0]), List.of("admin:bcrypt:" + newHash), new OpenOption[0]);
            logger.info("Password changed successfully for user: {}", (Object)DEFAULT_USERNAME);
            return ResponseEntity.ok((Object)"Password changed successfully");
        }
        catch (IOException e) {
            logger.error("Password change failed", (Throwable)e);
            return ResponseEntity.status((HttpStatusCode)HttpStatus.INTERNAL_SERVER_ERROR).body((Object)"Failed to change password");
        }
    }

    private Bucket createBucket() {
        Bandwidth limit = Bandwidth.builder().capacity(5L).refillGreedy(5L, Duration.ofMinutes(1L)).initialTokens(5L).build();
        return Bucket.builder().addLimit(limit).build();
    }

    public void resetBucket(String ip) {
        this.buckets.put(ip, this.createBucket());
    }

    public void resetAllBuckets() {
        this.buckets.clear();
        logger.info("All rate limit buckets have been reset");
    }

    private Bucket getBucket(String ip) {
        return this.buckets.computeIfAbsent(ip, k -> this.createBucket());
    }

    private void ensurePasswordFileExists() throws IOException {
        try {
            File file = new File(PASSWORD_FILE);
            if (!file.exists() || Files.size(file.toPath()) == 0L) {
                logger.info("Creating password file with default credentials");
                String hash = BCrypt.hashpw((String)DEFAULT_PASS, (String)BCrypt.gensalt());
                Files.write(file.toPath(), List.of("admin:bcrypt:" + hash), new OpenOption[0]);
            }
        }
        catch (IOException e) {
            logger.error("Failed to create password file", (Throwable)e);
            throw e;
        }
    }

    private String[] loadCredentials() throws IOException {
        try {
            this.ensurePasswordFileExists();
            String content = Files.readString(Path.of(PASSWORD_FILE, new String[0])).trim();
            String[] parts = content.split(":", 3);
            if (parts.length != 3 || !HASH_ALGORITHM.equals(parts[1])) {
                logger.warn("Invalid password file format, recreating");
                String hash = BCrypt.hashpw((String)DEFAULT_PASS, (String)BCrypt.gensalt());
                String newContent = "admin:bcrypt:" + hash;
                Files.write(Path.of(PASSWORD_FILE, new String[0]), List.of(newContent), new OpenOption[0]);
                return newContent.split(":", 3);
            }
            return parts;
        }
        catch (IOException e) {
            logger.error("Failed to load credentials", (Throwable)e);
            throw new IOException("Unable to access credentials", e);
        }
    }
}

