/*
 * Decompiled with CFR 0.152.
 */
package net.savignano.uptrust.config.features.keysource.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.savignano.uptrust.config.features.general.model.UptrustDto;
import net.savignano.uptrust.config.features.keysource.model.KeySourceDto;
import net.savignano.uptrust.config.features.keysource.model.PendingFileDeletion;
import net.savignano.uptrust.config.features.keysource.validation.KeySourceValidator;
import net.savignano.uptrust.config.infrastructure.password.ConfigPasswordService;
import net.savignano.uptrust.config.infrastructure.persistence.PendingFileDeletionRepository;
import net.savignano.uptrust.config.infrastructure.persistence.UptrustRepository;
import net.savignano.uptrust.config.infrastructure.storage.FileStorageService;
import net.savignano.uptrust.config.shared.exception.ConfigurationNotFoundException;
import net.savignano.uptrust.config.shared.exception.FileOperationException;
import net.savignano.uptrust.config.shared.exception.KeySourceNotFoundException;
import net.savignano.uptrust.config.shared.exception.PasswordEncryptionException;
import net.savignano.uptrust.config.shared.exception.ValidationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class KeySourceService {
    private static final Logger logger = LoggerFactory.getLogger(KeySourceService.class);
    private final UptrustRepository uptrustRepository;
    private final KeySourceValidator keySourceValidator;
    private final FileStorageService fileStorageService;
    private final PendingFileDeletionRepository pendingFileDeletionRepository;
    private final ConfigPasswordService passwordService;

    @Autowired
    public KeySourceService(UptrustRepository uptrustRepository, KeySourceValidator keySourceValidator, FileStorageService fileStorageService, PendingFileDeletionRepository pendingFileDeletionRepository, ConfigPasswordService passwordService) {
        this.uptrustRepository = uptrustRepository;
        this.keySourceValidator = keySourceValidator;
        this.fileStorageService = fileStorageService;
        this.pendingFileDeletionRepository = pendingFileDeletionRepository;
        this.passwordService = passwordService;
    }

    public List<KeySourceDto> getAllKeySources() {
        List keySources = this.uptrustRepository.findTopByOrderByUuidAsc().map(uptrust -> uptrust.getKeyManagement().getKeysources()).orElseThrow(() -> new ConfigurationNotFoundException("Cannot retrieve key sources - Uptrust configuration not found"));
        return keySources;
    }

    public KeySourceDto addKeySource(KeySourceDto keySource) {
        List validationErrors = this.keySourceValidator.validate(keySource);
        if (!validationErrors.isEmpty()) {
            throw new ValidationException("Key source", validationErrors);
        }
        boolean newFileUploaded = this.handleFileUpload(keySource, null);
        this.encryptPasswords(keySource, null);
        try {
            UptrustDto uptrust = (UptrustDto)this.uptrustRepository.findTopByOrderByUuidAsc().orElseThrow(() -> new ConfigurationNotFoundException("Cannot add key source - Uptrust configuration not found"));
            List keySources = uptrust.getKeyManagement().getKeysources();
            keySource.setPriority(Integer.valueOf(keySources.isEmpty() ? 0 : keySources.stream().mapToInt(ks -> ks.getPriority() != null ? ks.getPriority() : 0).max().orElse(-1) + 1));
            keySources.add(keySource);
            UptrustDto savedUptrust = (UptrustDto)this.uptrustRepository.save((Object)uptrust);
            logger.info("Added new key source with priority {}", (Object)keySource.getPriority());
            return savedUptrust.getKeyManagement().getKeysources().stream().filter(ks -> ks.getId().equals(keySource.getId())).findFirst().orElseThrow(() -> new RuntimeException("Failed to retrieve saved key source with ID: " + keySource.getId()));
        }
        catch (Exception e) {
            if (newFileUploaded) {
                this.deleteAssociatedFile(keySource);
            }
            throw e;
        }
    }

    public KeySourceDto updateKeySource(String uuid, KeySourceDto keySource) {
        UptrustDto uptrust = (UptrustDto)this.uptrustRepository.findTopByOrderByUuidAsc().orElseThrow(() -> new ConfigurationNotFoundException("Cannot update key source - Uptrust configuration not found"));
        List keySources = uptrust.getKeyManagement().getKeysources();
        KeySourceDto existingKeySource = keySources.stream().filter(ks -> ks.getUuid().equals(uuid)).findFirst().orElseThrow(() -> new KeySourceNotFoundException(uuid));
        List validationErrors = this.keySourceValidator.validate(keySource);
        if (!validationErrors.isEmpty()) {
            throw new ValidationException("Key source", validationErrors);
        }
        int index = keySources.indexOf(existingKeySource);
        boolean newFileUploaded = this.handleFileUpload(keySource, existingKeySource);
        this.encryptPasswords(keySource, existingKeySource);
        this.preserveEnvironmentSettings(keySource, existingKeySource);
        try {
            keySources.set(index, keySource);
            UptrustDto savedUptrust = (UptrustDto)this.uptrustRepository.save((Object)uptrust);
            logger.info("Updated key source with UUID: {}", (Object)uuid);
            return savedUptrust.getKeyManagement().getKeysources().stream().filter(ks -> ks.getId().equals(keySource.getId())).findFirst().orElseThrow();
        }
        catch (Exception e) {
            if (newFileUploaded) {
                this.deleteAssociatedFile(keySource);
            }
            throw e;
        }
    }

    public void deleteKeySource(String uuid) {
        UptrustDto uptrust = (UptrustDto)this.uptrustRepository.findTopByOrderByUuidAsc().orElseThrow(() -> new ConfigurationNotFoundException("Cannot delete key source - Uptrust configuration not found"));
        List keySources = uptrust.getKeyManagement().getKeysources();
        Optional<KeySourceDto> keySourceToDelete = keySources.stream().filter(ks -> ks.getUuid().equals(uuid)).findFirst();
        if (keySourceToDelete.isEmpty()) {
            throw new KeySourceNotFoundException(uuid);
        }
        this.deleteAssociatedFile(keySourceToDelete.get());
        keySources.remove(keySourceToDelete.get());
        this.uptrustRepository.save((Object)uptrust);
        logger.info("Deleted key source with UUID: {}", (Object)uuid);
    }

    private boolean handleFileUpload(KeySourceDto keySource, KeySourceDto existingKeySource) {
        boolean newFileUploaded = false;
        if (keySource.getConfig() != null && keySource.getConfig().getFileUpload() != null && !keySource.getConfig().getFileUpload().isEmpty()) {
            try {
                if (existingKeySource != null && existingKeySource.getConfig() != null && existingKeySource.getConfig().getFile() != null) {
                    this.deleteAssociatedFile(existingKeySource);
                }
                String storedFileName = this.fileStorageService.storeUploadedFile(keySource.getConfig().getFileUpload());
                keySource.getConfig().setFile(storedFileName);
                newFileUploaded = true;
            }
            catch (Exception e) {
                logger.error("Failed to handle file upload", (Throwable)e);
                throw new FileOperationException("File upload failed: " + e.getMessage(), (Throwable)e);
            }
        } else if (existingKeySource != null && existingKeySource.getConfig() != null) {
            keySource.getConfig().setFile(existingKeySource.getConfig().getFile());
        }
        return newFileUploaded;
    }

    private void preserveEnvironmentSettings(KeySourceDto keySource, KeySourceDto existingKeySource) {
        if (existingKeySource.getConfig() != null && existingKeySource.getConfig().getEnv() != null) {
            ArrayList existingEnv = new ArrayList(existingKeySource.getConfig().getEnv());
            keySource.getConfig().setEnv(existingEnv);
        }
    }

    public KeySourceDto moveKeySourceUp(String uuid) {
        UptrustDto uptrust = (UptrustDto)this.uptrustRepository.findTopByOrderByUuidAsc().orElseThrow(() -> new ConfigurationNotFoundException("Cannot move key source - Uptrust configuration not found"));
        List keySources = uptrust.getKeyManagement().getKeysources();
        int currentIndex = -1;
        for (int i = 0; i < keySources.size(); ++i) {
            if (!((KeySourceDto)keySources.get(i)).getUuid().equals(uuid)) continue;
            currentIndex = i;
            break;
        }
        if (currentIndex == -1) {
            throw new KeySourceNotFoundException(uuid);
        }
        if (currentIndex == 0) {
            throw new IllegalStateException("Key source is already at the top");
        }
        KeySourceDto currentKeySource = (KeySourceDto)keySources.get(currentIndex);
        KeySourceDto previousKeySource = (KeySourceDto)keySources.get(currentIndex - 1);
        int tempPriority = currentKeySource.getPriority();
        currentKeySource.setPriority(previousKeySource.getPriority());
        previousKeySource.setPriority(Integer.valueOf(tempPriority));
        UptrustDto savedUptrust = (UptrustDto)this.uptrustRepository.save((Object)uptrust);
        logger.info("Moved key source {} up", (Object)uuid);
        return savedUptrust.getKeyManagement().getKeysources().stream().filter(ks -> ks.getUuid().equals(uuid)).findFirst().orElseThrow();
    }

    public KeySourceDto moveKeySourceDown(String uuid) {
        UptrustDto uptrust = (UptrustDto)this.uptrustRepository.findTopByOrderByUuidAsc().orElseThrow(() -> new ConfigurationNotFoundException("Cannot move key source - Uptrust configuration not found"));
        List keySources = uptrust.getKeyManagement().getKeysources();
        int currentIndex = -1;
        for (int i = 0; i < keySources.size(); ++i) {
            if (!((KeySourceDto)keySources.get(i)).getUuid().equals(uuid)) continue;
            currentIndex = i;
            break;
        }
        if (currentIndex == -1) {
            throw new KeySourceNotFoundException(uuid);
        }
        if (currentIndex == keySources.size() - 1) {
            throw new IllegalStateException("Key source is already at the bottom");
        }
        KeySourceDto currentKeySource = (KeySourceDto)keySources.get(currentIndex);
        KeySourceDto nextKeySource = (KeySourceDto)keySources.get(currentIndex + 1);
        int tempPriority = currentKeySource.getPriority();
        currentKeySource.setPriority(nextKeySource.getPriority());
        nextKeySource.setPriority(Integer.valueOf(tempPriority));
        UptrustDto savedUptrust = (UptrustDto)this.uptrustRepository.save((Object)uptrust);
        logger.info("Moved key source {} down", (Object)uuid);
        return savedUptrust.getKeyManagement().getKeysources().stream().filter(ks -> ks.getUuid().equals(uuid)).findFirst().orElseThrow();
    }

    private void encryptPasswords(KeySourceDto keySource, KeySourceDto existingKeySource) {
        if (keySource.getConfig() == null) {
            return;
        }
        try {
            if (keySource.getConfig().getPassword() != null) {
                String existingEncryptedPassword = existingKeySource != null && existingKeySource.getConfig() != null ? existingKeySource.getConfig().getPassword() : null;
                String encryptedPassword = this.passwordService.encryptIfChanged(keySource.getConfig().getPassword(), existingEncryptedPassword);
                keySource.getConfig().setPassword(encryptedPassword);
                keySource.getConfig().setPasswordCipher(this.passwordService.getCipherType());
                logger.debug("Processed keystore password for key source");
            }
            if (keySource.getConfig().getAuth() != null && keySource.getConfig().getAuth().getPassword() != null) {
                String existingEncryptedAuthPassword = existingKeySource != null && existingKeySource.getConfig() != null && existingKeySource.getConfig().getAuth() != null ? existingKeySource.getConfig().getAuth().getPassword() : null;
                String encryptedAuthPassword = this.passwordService.encryptIfChanged(keySource.getConfig().getAuth().getPassword(), existingEncryptedAuthPassword);
                keySource.getConfig().getAuth().setPassword(encryptedAuthPassword);
                keySource.getConfig().getAuth().setPasswordCipher(this.passwordService.getCipherType());
                logger.debug("Processed auth password for key source");
            }
        }
        catch (Exception e) {
            logger.error("Failed to encrypt passwords for key source", (Throwable)e);
            throw new PasswordEncryptionException("Password encryption failed: " + e.getMessage(), (Throwable)e);
        }
    }

    private void deleteAssociatedFile(KeySourceDto keySource) {
        if (keySource.getConfig() != null && keySource.getConfig().getFile() != null) {
            try {
                this.pendingFileDeletionRepository.save((Object)PendingFileDeletion.builder().filePath(keySource.getConfig().getFile()).keySourceId(keySource.getId()).createdAt(LocalDateTime.now()).build());
                logger.info("Marked file for deletion: {}", (Object)keySource.getConfig().getFile());
            }
            catch (Exception e) {
                logger.warn("Failed to mark file for deletion for key source {}", (Object)keySource.getUuid(), (Object)e);
            }
        }
    }
}

