/*
 * Decompiled with CFR 0.152.
 */
package net.savignano.uptrust.proxy.base.handler.incoming;

import java.util.Collection;
import java.util.EnumSet;
import java.util.Map;
import java.util.Set;
import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import net.savignano.cryptography.enums.ECryptographyType;
import net.savignano.cryptography.enums.EValidationType;
import net.savignano.cryptography.key.ICryptographyKey;
import net.savignano.cryptography.util.MessageUtil;
import net.savignano.uptrust.proxy.base.handler.IMailHandler;
import net.savignano.uptrust.proxy.base.handler.incoming.IncomingMailHandlerConfiguration;
import net.savignano.uptrust.service.cryptography.IDecryptionService;
import net.savignano.uptrust.service.cryptography.IExtractionService;
import net.savignano.uptrust.service.cryptography.IValidationService;
import net.savignano.uptrust.service.decorator.IDecoratorService;
import net.savignano.uptrust.service.key.IKeyService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IncomingMailHandler
implements IMailHandler {
    private static final Logger LOG = LoggerFactory.getLogger(IncomingMailHandler.class);
    private static final String DECRYPTION_ERROR_HEADER = "X-DecryptionError";
    private static final String VALIDATION_ERROR_HEADER = "X-ValidationError";
    private final IncomingMailHandlerConfiguration config;

    public IncomingMailHandler(IncomingMailHandlerConfiguration config) {
        this.config = config;
        if (config == null) {
            throw new NullPointerException("Configuration must not be null.");
        }
    }

    @Override
    public void handleMessage(MimeMessage msg, Map<String, Object> context) {
        String msgId = MessageUtil.getMessageId((Message)msg);
        LOG.debug("Handling incoming message with ID \"{}\".", (Object)msgId);
        LOG.trace("Context for message handling: {}", context);
        if (!this.config.getLicenseService().isValid()) {
            LOG.error("No valid license configured. Message with ID \"{}\" will not be cryptographically processed.", (Object)msgId);
            return;
        }
        if (this.config.isClientSupportsProtection()) {
            LOG.debug("\"Client Supports Protection\" enabled. Message with ID \"{}\" will not be changed.", (Object)msgId);
            return;
        }
        EnumSet<IDecoratorService.EDecoration> decos = EnumSet.of(IDecoratorService.EDecoration.NO_CRYPTOGRAPHY, IDecoratorService.EDecoration.NOT_ENCRYPTED, IDecoratorService.EDecoration.NOT_SIGNED);
        int depth = 0;
        while (depth++ < 10 && this.dispatch(msg, decos)) {
        }
        this.decorate(msg, decos);
        if (!msgId.equals(MessageUtil.getMessageId((Message)msg))) {
            try {
                msg.setHeader("Message-ID", msgId);
            }
            catch (MessagingException e) {
                LOG.error("Could not reset message ID for message with sequence number: " + msg.getMessageNumber(), (Throwable)e);
            }
        }
    }

    private boolean dispatch(MimeMessage msg, Collection<IDecoratorService.EDecoration> decos) {
        IDecryptionService decService = this.config.getDecryptionService();
        if (decService != null && decService.isEncrypted(msg)) {
            return this.decrypt(decService, msg, decos);
        }
        IValidationService valService = this.config.getValidationService();
        if (valService != null && valService.isSigned(msg)) {
            return this.validate(valService, msg, decos);
        }
        return false;
    }

    private boolean decrypt(IDecryptionService service, MimeMessage msg, Collection<IDecoratorService.EDecoration> decos) {
        IDecryptionService.DecryptionResult result = service.decrypt(msg);
        LOG.info("Decryption result for message with ID \"{}\": {}", (Object)result.getMessageId(), (Object)result);
        decos.remove(IDecoratorService.EDecoration.NO_CRYPTOGRAPHY);
        decos.remove(IDecoratorService.EDecoration.NOT_ENCRYPTED);
        decos.add(IDecoratorService.EDecoration.fromCryptography((ECryptographyType)result.getCryptography()));
        if (result.isSuccess()) {
            decos.add(IDecoratorService.EDecoration.ENCRYPTED);
            return true;
        }
        if (result.getException() != null) {
            decos.add(IDecoratorService.EDecoration.ENCRYPTED_FAILURE);
            try {
                MessageUtil.setHeader((Message)msg, (String)DECRYPTION_ERROR_HEADER, (String)"Could not decrypt message. Error message: {0}", (Object[])new Object[]{result.getException().getMessage()});
            }
            catch (MessagingException e) {
                LOG.error("Could not set decryption error header in message with ID \"" + result.getMessageId() + "\". Error message: " + e.getMessage(), (Throwable)e);
            }
            return false;
        }
        if (result.isDecryptionKeyMissing()) {
            LOG.info("Message with ID \"{}\" could not get decrypted because no matching key for decryption was available.", (Object)result.getMessageId());
            decos.add(IDecoratorService.EDecoration.ENCRYPTED_UNKNOWN);
            return false;
        }
        LOG.debug("Implementation error. Fix me.");
        decos.add(IDecoratorService.EDecoration.ENCRYPTED_FAILURE);
        return false;
    }

    private boolean validate(IValidationService service, MimeMessage msg, Collection<IDecoratorService.EDecoration> decos) {
        boolean useExtractedKey;
        IExtractionService.ExtractionResult extractionResult = this.extractKey(this.config.getExtractionService(), msg);
        IValidationService.ValidationResult result = service.validate(msg);
        LOG.info("Validation result for message with ID \"{}\": {}", (Object)result.getMessageId(), (Object)result);
        decos.remove(IDecoratorService.EDecoration.NO_CRYPTOGRAPHY);
        decos.remove(IDecoratorService.EDecoration.NOT_SIGNED);
        decos.add(IDecoratorService.EDecoration.fromCryptography((ECryptographyType)result.getCryptography()));
        if (result.isSuccess()) {
            useExtractedKey = true;
            decos.add(IDecoratorService.EDecoration.SIGNED);
        } else if (result.getException() != null) {
            useExtractedKey = false;
            decos.add(IDecoratorService.EDecoration.SIGNED_FAILURE);
            try {
                MessageUtil.setHeader((Message)msg, (String)VALIDATION_ERROR_HEADER, (String)"Could not validate message. Error message: {0}", (Object[])new Object[]{result.getException().getMessage()});
            }
            catch (MessagingException e) {
                LOG.error("Could not set validation error header in message with ID \"" + result.getMessageId() + "\". Error message: " + e.getMessage(), (Throwable)e);
            }
        } else if (!this.config.isCheckTrust()) {
            EnumSet<EValidationType> toCheck = EnumSet.copyOf(result.getExpectedValidations());
            toCheck.remove(EValidationType.TRUST);
            useExtractedKey = toCheck.equals(result.getActualValidations());
            decos.add(this.checkValidationResult(result.getActualValidations(), toCheck, msg));
        } else {
            useExtractedKey = false;
            decos.add(this.checkValidationResult(result.getActualValidations(), result.getExpectedValidations(), msg));
        }
        if (useExtractedKey && extractionResult.isSuccess()) {
            this.storeKey(this.config.getKeyService(), extractionResult);
        }
        return false;
    }

    private IExtractionService.ExtractionResult extractKey(IExtractionService service, MimeMessage msg) {
        if (service == null) {
            return new IExtractionService.ExtractionResult();
        }
        IExtractionService.ExtractionResult result = service.extract(msg);
        LOG.info("Extraction result for message with ID \"{}\": {}", (Object)result.getMessageId(), (Object)result);
        return result;
    }

    private void storeKey(IKeyService service, IExtractionService.ExtractionResult result) {
        if (service == null) {
            return;
        }
        String email = MessageUtil.getEmail((Address)result.getAddress());
        service.setKey((Object)email, (ICryptographyKey)result.getKey().get());
    }

    private IDecoratorService.EDecoration checkValidationResult(Set<EValidationType> result, Set<EValidationType> checkFor, MimeMessage msg) {
        IDecoratorService.EDecoration deco = null;
        String validationErrorHeader = null;
        String msgId = MessageUtil.getMessageId((Message)msg);
        if (result.containsAll(checkFor)) {
            LOG.debug("Message with ID \"{}\" is correctly signed.", (Object)msgId);
            deco = IDecoratorService.EDecoration.SIGNED;
        } else if (checkFor.contains(EValidationType.CAN_VALIDATE) && !result.contains(EValidationType.CAN_VALIDATE)) {
            LOG.info("Message with ID \"{}\" has an unknown sender, cannot verify.", (Object)msgId);
            validationErrorHeader = "Unknown sender.";
            deco = IDecoratorService.EDecoration.SIGNED_UNVERIFIED;
        } else if (checkFor.contains(EValidationType.SIGNATURE) && !result.contains(EValidationType.SIGNATURE)) {
            LOG.warn("Validity of message with ID \"{}\" could not be verified. Message was probably tampered with.", (Object)msgId);
            validationErrorHeader = "Signature wrong.";
            deco = IDecoratorService.EDecoration.SIGNED_MISMATCH;
        } else if (checkFor.contains(EValidationType.TRUST) && !result.contains(EValidationType.TRUST)) {
            LOG.warn("Trust of signer of message with ID \"{}\" could not be verified.", (Object)msgId);
            validationErrorHeader = "Untrusted sender.";
            deco = IDecoratorService.EDecoration.SIGNED_UNVERIFIED;
        } else {
            LOG.error("Unknown problem with signature for message with ID \"{}\". Check for: {}. Result: {}", new Object[]{msgId, checkFor, result});
            deco = IDecoratorService.EDecoration.SIGNED_FAILURE;
        }
        if (validationErrorHeader != null) {
            try {
                MessageUtil.setHeader((Message)msg, (String)VALIDATION_ERROR_HEADER, validationErrorHeader);
            }
            catch (MessagingException e) {
                LOG.error("Could not set header in message with ID \"" + msgId + "\". Error message: " + e.getMessage(), (Throwable)e);
            }
        }
        if (deco == null) {
            LOG.warn("Implementation error when determining decorations for message with ID \"{}\". Fix me.", (Object)msgId);
            deco = IDecoratorService.EDecoration.SIGNED_FAILURE;
        }
        return deco;
    }

    private void decorate(MimeMessage msg, Collection<IDecoratorService.EDecoration> decos) {
        IDecoratorService decoService = this.config.getDecoratorService();
        if (decoService != null) {
            IDecoratorService.DecorationResult result = decoService.decorate(msg, decos);
            LOG.info("Decoration result for message with ID \"{}\": {}", (Object)result.getMessageId(), (Object)result);
        }
    }
}

