/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.transport.mailets.remote.delivery;

import com.github.fge.lambdas.Throwing;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.SendFailedException;
import javax.mail.internet.InternetAddress;
import org.apache.james.core.Domain;
import org.apache.james.core.MailAddress;
import org.apache.james.dnsservice.api.DNSService;
import org.apache.james.dnsservice.api.TemporaryResolutionException;
import org.apache.james.lifecycle.api.LifecycleUtil;
import org.apache.james.transport.mailets.remote.delivery.AddressesArrayToMailAddressListConverter;
import org.apache.james.transport.mailets.remote.delivery.Bouncer;
import org.apache.james.transport.mailets.remote.delivery.DeliveryRetriesHelper;
import org.apache.james.transport.mailets.remote.delivery.DnsHelper;
import org.apache.james.transport.mailets.remote.delivery.EnhancedMessagingException;
import org.apache.james.transport.mailets.remote.delivery.ExecutionResult;
import org.apache.james.transport.mailets.remote.delivery.InternetAddressConverter;
import org.apache.james.transport.mailets.remote.delivery.MailDelivrerToHost;
import org.apache.james.transport.mailets.remote.delivery.MessageComposer;
import org.apache.james.transport.mailets.remote.delivery.RemoteDeliveryConfiguration;
import org.apache.mailet.HostAddress;
import org.apache.mailet.Mail;
import org.apache.mailet.MailetContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MailDelivrer {
    private static final Logger LOGGER = LoggerFactory.getLogger(MailDelivrer.class);
    private final RemoteDeliveryConfiguration configuration;
    private final MailDelivrerToHost mailDelivrerToHost;
    private final DnsHelper dnsHelper;
    private final MessageComposer messageComposer;
    private final Bouncer bouncer;
    private final MailetContext mailetContext;

    public MailDelivrer(RemoteDeliveryConfiguration configuration, MailDelivrerToHost mailDelivrerToHost, DNSService dnsServer, Bouncer bouncer, MailetContext mailetContext) {
        this(configuration, mailDelivrerToHost, new DnsHelper(dnsServer, configuration), bouncer, mailetContext);
    }

    @VisibleForTesting
    MailDelivrer(RemoteDeliveryConfiguration configuration, MailDelivrerToHost mailDelivrerToHost, DnsHelper dnsHelper, Bouncer bouncer, MailetContext mailetContext) {
        this.configuration = configuration;
        this.mailDelivrerToHost = mailDelivrerToHost;
        this.dnsHelper = dnsHelper;
        this.messageComposer = new MessageComposer(configuration);
        this.bouncer = bouncer;
        this.mailetContext = mailetContext;
    }

    public ExecutionResult deliver(Mail mail) {
        try {
            return this.tryDeliver(mail);
        }
        catch (SendFailedException sfe) {
            return this.handleSenderFailedException(mail, sfe);
        }
        catch (MessagingException ex) {
            boolean isPermanent = new EnhancedMessagingException(ex).isServerError();
            return this.logAndReturn(mail, ExecutionResult.onFailure(isPermanent, (Exception)((Object)ex)));
        }
        catch (Exception ex) {
            LOGGER.error("Generic exception = permanent failure: {}", (Object)ex.getMessage(), (Object)ex);
            return this.logAndReturn(mail, ExecutionResult.permanentFailure(ex));
        }
    }

    private ExecutionResult tryDeliver(Mail mail) throws MessagingException {
        if (mail.getRecipients().isEmpty()) {
            LOGGER.info("No recipients specified... not sure how this could have happened.");
            return ExecutionResult.permanentFailure(new Exception("No recipients specified for " + mail.getName() + " sent by " + mail.getMaybeSender().asString()));
        }
        if (this.configuration.isDebug()) {
            LOGGER.debug("Attempting to deliver {}", (Object)mail.getName());
        }
        Domain host = this.retrieveTargetHostname(mail);
        try {
            Iterator<HostAddress> targetServers = this.dnsHelper.retrieveHostAddressIterator(host.asString(), this.configuration.isSSLEnable());
            if (!targetServers.hasNext()) {
                return this.handleNoTargetServer(mail, host);
            }
            return this.doDeliver(mail, (Set<InternetAddress>)InternetAddressConverter.convert(mail.getRecipients()), targetServers);
        }
        catch (TemporaryResolutionException e) {
            return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)new MessagingException("Temporary problem looking up mail server for host: " + String.valueOf(host) + ".  I cannot determine where to send this message."))));
        }
    }

    private Domain retrieveTargetHostname(Mail mail) {
        Preconditions.checkArgument((!mail.getRecipients().isEmpty() ? 1 : 0) != 0, (Object)"Mail should have recipients to attempt delivery");
        MailAddress rcpt = (MailAddress)Iterables.getFirst((Iterable)mail.getRecipients(), null);
        return rcpt.getDomain();
    }

    private ExecutionResult doDeliver(Mail mail, Set<InternetAddress> addr, Iterator<HostAddress> targetServers) throws MessagingException {
        MessagingException lastError = null;
        HashSet<InternetAddress> targetAddresses = new HashSet<InternetAddress>(addr);
        while (targetServers.hasNext()) {
            try {
                return this.mailDelivrerToHost.tryDeliveryToHost(mail, targetAddresses, targetServers.next());
            }
            catch (SendFailedException sfe) {
                lastError = this.handleSendFailExceptionOnMxIteration(mail, sfe);
                ImmutableList<InternetAddress> deliveredAddresses = this.listDeliveredAddresses(sfe);
                this.configuration.getOnSuccess().ifPresent((Consumer<String>)Throwing.consumer(onSuccess -> {
                    Mail copy = mail.duplicate();
                    try {
                        copy.setRecipients((Collection)deliveredAddresses.stream().map(Throwing.function(MailAddress::new)).collect(ImmutableList.toImmutableList()));
                        this.mailetContext.sendMail(copy, onSuccess);
                    }
                    finally {
                        LifecycleUtil.dispose((Object)copy);
                    }
                }));
                targetAddresses.removeAll((Collection<?>)deliveredAddresses);
            }
            catch (MessagingException me) {
                lastError = this.handleMessagingException(mail, me);
                if (this.configuration.isDebug()) {
                    LOGGER.debug(me.getMessage(), me.getCause());
                    continue;
                }
                LOGGER.info(me.getMessage());
            }
        }
        if (lastError != null) {
            throw lastError;
        }
        return ExecutionResult.temporaryFailure();
    }

    private ImmutableList<InternetAddress> listDeliveredAddresses(SendFailedException sfe) {
        return Optional.ofNullable(sfe.getValidSentAddresses()).map(addresses -> (ImmutableList)Arrays.stream(addresses).map(InternetAddress.class::cast).collect(ImmutableList.toImmutableList())).orElse(ImmutableList.of());
    }

    private MessagingException handleMessagingException(Mail mail, MessagingException me) throws MessagingException {
        LOGGER.debug("Exception delivering message ({}) - {}", (Object)mail.getName(), (Object)me.getMessage());
        if (me.getNextException() instanceof IOException) {
            return me;
        }
        throw me;
    }

    @VisibleForTesting
    ExecutionResult handleSenderFailedException(Mail mail, SendFailedException sfe) {
        this.logSendFailedException(sfe);
        EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe));
        List<MailAddress> invalidAddresses = AddressesArrayToMailAddressListConverter.getAddressesAsMailAddress(sfe.getInvalidAddresses());
        List<MailAddress> validUnsentAddresses = AddressesArrayToMailAddressListConverter.getAddressesAsMailAddress(sfe.getValidUnsentAddresses());
        if (this.configuration.isDebug()) {
            LOGGER.debug("Mail {} has initially recipients: {}", (Object)mail.getName(), (Object)mail.getRecipients());
            if (!invalidAddresses.isEmpty()) {
                LOGGER.debug("Invalid recipients: {}", invalidAddresses);
            }
            if (!validUnsentAddresses.isEmpty()) {
                LOGGER.debug("Unsent recipients: {}", validUnsentAddresses);
            }
        }
        if (!validUnsentAddresses.isEmpty()) {
            if (!invalidAddresses.isEmpty()) {
                mail.setRecipients(invalidAddresses);
                this.bouncer.bounce(mail, (Exception)((Object)sfe));
            }
            mail.setRecipients(validUnsentAddresses);
            if (enhancedMessagingException.hasReturnCode()) {
                boolean isPermanent = enhancedMessagingException.isServerError();
                return this.logAndReturn(mail, ExecutionResult.onFailure(isPermanent, (Exception)((Object)sfe)));
            }
            return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)sfe)));
        }
        if (!invalidAddresses.isEmpty()) {
            mail.setRecipients(invalidAddresses);
            return this.logAndReturn(mail, ExecutionResult.permanentFailure((Exception)((Object)sfe)));
        }
        if ((enhancedMessagingException.hasReturnCode() || enhancedMessagingException.hasNestedReturnCode()) && enhancedMessagingException.isServerError()) {
            return ExecutionResult.permanentFailure((Exception)((Object)sfe));
        }
        return ExecutionResult.temporaryFailure((Exception)((Object)sfe));
    }

    private ExecutionResult logAndReturn(Mail mail, ExecutionResult executionResult) {
        LOGGER.debug(this.messageComposer.composeFailLogMessage(mail, executionResult));
        return executionResult;
    }

    private MessagingException handleSendFailExceptionOnMxIteration(Mail mail, SendFailedException sfe) throws SendFailedException {
        EnhancedMessagingException enhancedMessagingException;
        Address[] validSent;
        this.logSendFailedException(sfe);
        if (sfe.getValidSentAddresses() != null && (validSent = sfe.getValidSentAddresses()).length > 0) {
            LOGGER.debug("Mail ({}) sent successfully for {}", (Object)mail.getName(), (Object)validSent);
        }
        if ((enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe))).isServerError()) {
            throw sfe;
        }
        Address[] validUnsentAddresses = sfe.getValidUnsentAddresses();
        if (validUnsentAddresses != null && validUnsentAddresses.length > 0) {
            if (this.configuration.isDebug()) {
                LOGGER.debug("Send failed, {} valid addresses remain, continuing with any other servers", (Object)validUnsentAddresses);
            }
            return sfe;
        }
        throw sfe;
    }

    private ExecutionResult handleNoTargetServer(Mail mail, Domain host) {
        LOGGER.info("No mail server found for: {}", (Object)host.name());
        MessagingException messagingException = new MessagingException("There are no DNS entries for the hostname " + String.valueOf(host) + ".  I cannot determine where to send this message.");
        int retry = DeliveryRetriesHelper.retrieveRetries(mail);
        if (retry >= this.configuration.getDnsProblemRetry()) {
            return this.logAndReturn(mail, ExecutionResult.permanentFailure((Exception)((Object)messagingException)));
        }
        return this.logAndReturn(mail, ExecutionResult.temporaryFailure((Exception)((Object)messagingException)));
    }

    private void logSendFailedException(SendFailedException sfe) {
        if (this.configuration.isDebug()) {
            EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException((MessagingException)((Object)sfe));
            if (enhancedMessagingException.hasReturnCode()) {
                LOGGER.info("SMTP SEND FAILED: Command [{}] RetCode: [{}] Response[{}]", new Object[]{enhancedMessagingException.computeCommand(), enhancedMessagingException.getReturnCode(), sfe.getMessage()});
            } else {
                LOGGER.info("Send failed", (Throwable)sfe);
            }
            this.logLevels((MessagingException)((Object)sfe));
        }
    }

    private void logLevels(MessagingException me) {
        Exception ne;
        while ((ne = me.getNextException()) != null && ne instanceof MessagingException) {
            me = (MessagingException)((Object)ne);
            EnhancedMessagingException enhancedMessagingException = new EnhancedMessagingException(me);
            if (!((Object)((Object)me)).getClass().getName().endsWith(".SMTPAddressFailedException") && !((Object)((Object)me)).getClass().getName().endsWith(".SMTPAddressSucceededException")) continue;
            LOGGER.debug("ADDRESS :[{}] Address:[{}] Command : [{}] RetCode[{}] Response [{}]", new Object[]{enhancedMessagingException.computeAction(), me, enhancedMessagingException.computeAddress(), enhancedMessagingException.computeCommand(), enhancedMessagingException.getReturnCode()});
        }
    }
}

